from flask import Blueprint, render_template, request, redirect, url_for, flash, send_file, current_app
from werkzeug.utils import secure_filename
import os
import pandas as pd
import random
import io
from itertools import combinations
from app.utils import allowed_file

bp = Blueprint('main', __name__)

def create_pools(teams, num_pools, teams_per_pool):
    """Randomly distribute teams into pools."""
    random.shuffle(teams)
    return [teams[i:i + teams_per_pool] for i in range(0, len(teams), teams_per_pool)]

def schedule_matches(pools):
    """
    Generate all possible matches within each pool
    
    Args:
        pools: Dictionary {
            "Pool 1": ["Team A", "Team B"],
            "Pool 2": ["Team C", "Team D"]
        }
    
    Returns:
        Dictionary {
            "Pool 1": [("Team A", "Team B")],
            "Pool 2": [("Team C", "Team D")]
        }
    """
    schedule = {}
    
    for pool_name, teams in pools.items():
        # Generate all unique match combinations
        matches = list(combinations(teams, 2))
        
        # Add to schedule dictionary
        schedule[pool_name] = matches
    
    return schedule

@bp.route('/', methods=['GET', 'POST'])
def index():
    """Handle file upload and team selection."""
    if request.method == 'POST':
        if 'file' not in request.files:
            flash('No file selected')
            return redirect(request.url)
        
        file = request.files['file']
        if file.filename == '':
            flash('No file selected')
            return redirect(request.url)
            
        if file and allowed_file(file.filename):
            filename = secure_filename(file.filename)
            upload_dir = current_app.config['UPLOAD_FOLDER']
            
            # Ensure upload directory exists
            os.makedirs(upload_dir, exist_ok=True)
            
            filepath = os.path.join(upload_dir, filename)
            try:
                file.save(filepath)
                
                # Read file based on extension
                if filename.endswith('.csv'):
                    df = pd.read_csv(filepath)
                else:  # Assume Excel
                    df = pd.read_excel(filepath)
                
                teams = df.iloc[:, 0].dropna().tolist()
                return render_template('index.html', 
                                    teams=teams, 
                                    filename=filename)
                
            except Exception as e:
                flash(f'Error processing file: {str(e)}')
                if os.path.exists(filepath):
                    os.remove(filepath)
                return redirect(request.url)
    
    return render_template('index.html')

@bp.route('/generate', methods=['POST'])
def generate():
    try:
        teams = request.form.getlist('teams[]')
        num_pools = int(request.form['num_pools'])
        teams_per_pool = int(request.form['teams_per_pool'])
        
        if len(teams) % teams_per_pool != 0:
            flash('Teams cannot be divided evenly into pools. Adjust teams per pool.')
            return redirect(url_for('main.index'))
        
        # Create pools as a dictionary for easy template rendering
        pools = {}
        random.shuffle(teams)
        
        for i in range(num_pools):
            pool_num = i + 1
            start_idx = i * teams_per_pool
            end_idx = start_idx + teams_per_pool
            pools[f"Pool {pool_num}"] = teams[start_idx:end_idx]
        
        return render_template('results.html', 
                            pools=pools,
                            match_schedule=schedule_matches(pools))
        
    except Exception as e:
        flash(f'Error generating schedule: {str(e)}')
        return redirect(url_for('main.index'))

@bp.route('/export', methods=['POST'])
def export():
    """Export schedule to Excel."""
    try:
        pools = request.form.getlist('pools[]')
        matches = request.form.getlist('matches[]')
        
        # Reconstruct data structure safely
        data = []
        for pool in pools:
            pool_matches = [m.split(' vs ') for m in matches if m.startswith(pool)]
            data.extend([[pool, m[0], m[1]] for m in pool_matches])
        
        df = pd.DataFrame(data, columns=["Pool", "Team 1", "Team 2"])
        
        output = io.BytesIO()
        df.to_excel(output, index=False, engine='openpyxl')
        output.seek(0)
        
        return send_file(
            output,
            mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            as_attachment=True,
            download_name='tournament_schedule.xlsx'
        )
        
    except Exception as e:
        flash(f'Export failed: {str(e)}')
        return redirect(url_for('main.index'))